% ***********************************************
% An Introduction to Computational Macroeconomics
% A. Bongers, T. Gmez and J. L. Torres (2019)
% Dynamic IS-LM model in MATLAB
% File: m2.m
% ***********************************************
clear all
% Periods
T       =   30;
% Exogenous variables
Beta0   =   2100;
m0  	=   100;
ypot0   =   2000;
% Parameters
Theta   =   0.5;
Psi     =   0.01;
Beta1   =   50;
Mi      =   0.01;
Ni      =   0.2;
% Matrices
A=[0 Mi;-Ni*Beta1/Theta Ni*(Beta1*Mi-Beta1*Psi/Theta-1)];
B=[0 0 -Mi; Ni Ni*Beta1/Theta -Ni*Beta1*Mi];
Z=[Beta0 m0 ypot0]; 
% Steady State
pbar    =   (Theta*Beta0)/Beta1+m0-(Psi+Theta/Beta1)*ypot0;
ybar	=   ypot0;
dp(1)   =   0;
dy(1)   =   0;
p(1)    =   pbar;
y(1)    =   ybar;
i(1)    =   -(1/Theta)*(m0-p(1)-Psi*y(1));
yd(1)   =   Beta0-Beta1*(i(1)-dp(1));
% Eigenvalues
v=eig(A);
Lambda1=v(1);
Lambda2=v(2);
% Shock
m1=   101;
% The model
for j=1:T-1;
    y(j+1)  =   y(j)+dy(j);
    p(j+1)  =   p(j)+dp(j);
    dp(j+1) =   Mi*(y(j+1)-ybar);
    i(j+1)  =   -(1/Theta)*(m1-p(j+1)-Psi*y(j+1));
    yd(j+1) =   Beta0-Beta1*(i(j+1)-dp(j+1));
    dy(j+1) =   Ni*(yd(j+1)-y(j+1));
end;  
% Graphics
j=1:T;
subplot(2,2,1)
plot(j,y,'Color',[0.25 0.25 0.25],'linewidth',3.5)
xlabel('Periods','FontName','Times')
title('Output','FontName','Times')
axis([0 T min(y) max(y)])
subplot(2,2,2)
plot(j,yd,'Color',[0.25 0.25 0.25],'linewidth',3.5)
xlabel('Periods','FontName','Times')
title('Aggregate demand','FontName','Times')
axis([0 T min(yd) max(yd)])
subplot(2,2,3)
plot(j,p,'Color',[0.25 0.25 0.25],'linewidth',3.5)
xlabel('Periods','FontName','Times')
title('Prices','FontName','Times')
axis([0 T min(p) max(p)])
subplot(2,2,4)
plot(j,i,'Color',[0.25 0.25 0.25],'linewidth',3.5)
xlabel('Periods','FontName','Times')
title('Nominal interest rate','FontName','Times')
axis([0 T min(i) max(i)])
% Phase diagram
syms p y;
ymin=1980;
ymax=2040;
pmin=80;
pmax=84;
[y p]=meshgrid(ymin:(ymax-ymin)/20:ymax,pmin:(pmax-pmin)/20:pmax);
dp=Mi*(y-ypot0);
dy=Ni*(Beta0-Beta1*(-(1/Theta)*(m0-p-Psi*y)-Mi*(y-ypot0))-y);
figure;
quiver(y,p,dy,dp);
title('Phase diagram: Dynamic IS-LM model','FontName','Times')
xlabel('Output','FontName','Times')
ylabel('Prices','FontName','Times')
hold on
plot(ybar,pbar,'o','Color',[0 0 0])
% Path to the Steady State
dp=Mi*(y-ypot0);
dy=Ni*(Beta0-Beta1*(-(1/Theta)*(m1-p-Psi*y)-Mi*(y-ypot0))-y);
figure
quiver(y,p,dy,dp);
title('Dynamic IS-LM model: Path to the Steady State','FontName','Times')
xlabel('Output','FontName','Times')
ylabel('Prices','FontName','Times')
hold on
m=[ybar; pbar];
n=100;
d=@(y,p) [Ni*(Beta0-Beta1*(-(1/Theta)*(m1-p-Psi*y))-Mi*(y-ypot0)-y); Mi*(y-ypot0)];
for i=1:n
    mm = m+d(m(1),m(2));
    plot([m(1),mm(1)],[m(2),mm(2)],'o-','Color',[0 0 0])
    m=mm;
end